const { Sequelize } = require('sequelize');

const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const CoursesModel = db.model.courses;
exports.CoursesModel = CoursesModel;
const CourseCategoryModel = db.model.courseCategory;
const UserModel = db.model.user;
const CourseRegistration = db.model.courseRegistration;

exports.create = async (req, res) => {
    try {
        const {
            title,
            course_cat,
            date,
            price,
            discounted_price,
            instructor_id,
            description,
            status,
            features,
            duration,
        } = req.body;
        const { files } = req;

        const img = files?.find((file) => file?.fieldname === 'img')?.filename;

        const courseData = {
            title,
            course_cat,
            date,
            price,
            discounted_price,
            instructor_id: instructor_id || null,
            description,
            features: Array.isArray(features) ? features?.join(', ') : features,
            img,
            status,
            duration,
        };

        const result = await CoursesModel.create(courseData);

        if (!result) {
            return errorResponse(404, 'FAILED', 'service created failed', res);
        }

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// get all

// Fetch the course categories matching the IDs in the array

async function findCourseCatData(arrIds) {
    try {
        const courseCategory = await CourseCategoryModel.findAll({
            where: {
                id: {
                    [Sequelize.Op.in]: arrIds,
                },
            },

            attributes: ['id', 'title'],
        });

        return courseCategory;
    } catch (err) {
        console.log(err);
        return [];
    }
}
exports.findCourseCatData = findCourseCatData;

exports.getAll = async (_, res) => {
    try {
        const result = await CoursesModel.findAll();

        const courseCatArr = result.reduce((acc, cur) => {
            const catArr = cur.course_cat ? cur.course_cat.split(',') : [];
            catArr.forEach((catId) => {
                if (!acc.includes(catId)) {
                    acc.push(catId);
                }
            });
            return acc;
        }, []);

        const courseCategory = await findCourseCatData(courseCatArr);

        const dataWithCat = result.map((item) => {
            const catIds = item.course_cat ? item.course_cat.split(',') : [];
            const catArrData = catIds.map((catId) =>
                courseCategory.find((cat) => cat.id === Number(catId))
            );
            return {
                ...item.dataValues,
                course_category: catArrData,
            };
        });

        return successResponse(200, 'OK', dataWithCat, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server side problem', res);
    }
};

// get one

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;
        const { status } = req.query;
        const { user } = req;

        const whereCondition = {
            id,
        };

        if (status === 'active') {
            whereCondition.status = 1;
        }

        // Fetch the course entry
        const result = await CoursesModel.findOne({
            where: whereCondition,
            include: [
                {
                    model: UserModel,
                    as: 'instructor',
                    attributes: [
                        'id',
                        'f_name',
                        'l_name',
                        'email',
                        'mobile',
                        'image',
                        'role_id',
                        'remarks',
                    ],
                },
            ],
        });

        // If the course is not found, return a 404 response
        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        // Split the course_cat string into an array
        const courseCatArr = result.course_cat ? result.course_cat.split(',') : [];

        // Fetch the course categories matching the IDs in the array
        const courseCategory = await CourseCategoryModel.findAll({
            where: {
                id: {
                    [Sequelize.Op.in]: courseCatArr,
                },
            },

            attributes: ['id', 'title'],
        });

        // Add the retrieved categories to the course result
        result.dataValues.course_category = courseCategory;
        // result.dataValues.features = result.dataValues.features?.split(', ');

        result.dataValues.course_category = courseCategory;
        // result.dataValues.features = result.dataValues.features?.split(', ');

        // now lets get if this user has registred this course.

        if (user) {
            const regData = await CourseRegistration.findOne({
                where: {
                    course_id: id,
                    mobile: user?.mobile,
                },
            });
            result.dataValues.has_registred = !!regData;
        }

        // Return the success response with the course and its categories
        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

// update

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const {
            title,
            course_cat,
            date,
            price,
            discounted_price,
            instructor_id,
            description,
            status,
            features,
            duration,
        } = req.body;

        console.log(req.body);

        const { files } = req;

        const isExist = await CoursesModel.findOne({
            where: {
                id,
            },
        });

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No Data found with this id', res);
        }

        const img = files?.find((file) => file?.fieldname === 'img')?.filename || isExist?.img;

        const courseData = {
            title,
            course_cat,
            date,
            price,
            discounted_price,
            instructor_id,
            description,
            features: Array.isArray(features) ? features?.join(', ') : features,
            img,
            status,
            duration,
        };
        const result = await CoursesModel.update(courseData, {
            where: {
                id,
            },
        });

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Courses updated failed', res);
        }

        return successResponse(201, 'OK', 'Update Successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// delete

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await CoursesModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Hero section modal deleted failed', res);
        }

        return successResponse(204, 'OK', 'Hero section deleted successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server side error', res);
    }
};
