const { Sequelize, Op } = require("sequelize");

const db = require("../../../../config/database.config");

const errorResponse = require("../../../../utils/errorResponse");

const successResponse = require("../../../../utils/successResponse");

const EventCatModel = db.model.event_category;

const EventModel = db.model.events;

exports.create = async (req, res) => {
  try {
    const { title, info, order, status } = req.body;

    const eventCat = {
      title,
      info,
      order,
      status,
    };
    const result = await EventCatModel.create(eventCat);

    if (!result) {
      return errorResponse(404, "FAILED", "event category created failed", res);
    }

    return successResponse(201, "OK", result, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err?.message || "There is a server side problem",
      res,
    );
  }
};

// get all

exports.getAll = async (_, res) => {
  try {
    const result = await EventCatModel.findAll({
      order: [["order", "ASC"]],
    });

    return successResponse(200, "OK", result, res);
  } catch (error) {
    return errorResponse(
      500,
      "ERROR",
      error.message || "There is a server side problem",
      res,
    );
  }
};

// get one

exports.getOne = async (req, res) => {
  try {
    const { id } = req.params;
    const result = await EventCatModel.findOne({
      where: {
        id,
      },
    });

    if (!result) {
      return errorResponse(404, "NOT FOUND", "NO Active data found", res);
    }

    return successResponse(200, "OK", result, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err.message || "There was a server side Error",
      res,
    );
  }
};

exports.getActiveData = async (_, res) => {
  try {
    const result = await EventCatModel.findAll({
      where: {
        status: 1,
      },
    });

    const data = await Promise.all(
      result.map(async (category) => {
        const events = await EventModel.findAll({
          where: {
            event_cat: {
              [Op.like]: `%${category.id}%`,
            },
          },
        });

        return {
          ...category.dataValues,
          event_list: events,
        };
      }),
    );

    if (!result) {
      return errorResponse(404, "NOT FOUND", "NO Active data found", res);
    }

    return successResponse(200, "OK", data, res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err.message || "There was a server side Error",
      res,
    );
  }
};

// update

exports.update = async (req, res) => {
  try {
    const { id } = req.params;
    const { title, info, order, status } = req.body;

    const isExist = await EventCatModel.findOne({
      where: {
        id,
      },
    });

    if (!isExist) {
      return errorResponse(404, "NOT FOUND", "No Data found with this id", res);
    }

    const eventData = {
      title,
      info,
      order,
      status,
    };
    const result = await EventCatModel.update(eventData, {
      where: {
        id,
      },
    });

    if (!result.includes(1)) {
      return errorResponse(404, "FAILED", "Events Data created failed", res);
    }

    return successResponse(201, "OK", "Update Successfully", res);
  } catch (err) {
    return errorResponse(
      500,
      "ERROR",
      err?.message || "There is a server side problem",
      res,
    );
  }
};

// delete

exports.delete = async (req, res) => {
  try {
    const { id } = req.params;

    const result = await EventCatModel.destroy({
      where: {
        id,
      },
    });

    if (result !== 1) {
      return errorResponse(
        404,
        "FAILED",
        "Event Cat modal deleted failed",
        res,
      );
    }

    return successResponse(204, "OK", "Event Cat deleted successfully", res);
  } catch (err) {
    return errorResponse(500, "ERROR", "There is a server side error", res);
  }
};
