const { Sequelize } = require('sequelize');

const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const EventsModel = db.model.events;
exports.EventsModel = EventsModel;
const EventCategoryModel = db.model.event_category;
const { generateRRule } = require('../../../../utils/rruleGenerator');

exports.create = async (req, res) => {
    try {
        const {
            title,
            event_cat,
            date_type,
            event_date,
            event_date_time,
            short_description,
            description,
            status,
            location,
            order,
        } = req.body;
        const { files } = req;

        const event_img = files?.find((file) => file?.fieldname === 'event_img')?.filename;

        const eventData = {
            title,
            event_cat,
            date_type,
            event_date,
            event_date_time,
            event_img,
            short_description,
            description,
            status,
            location,
            order,
        };

        const result = await EventsModel.create(eventData);

        if (!result) {
            return errorResponse(404, 'FAILED', 'event created failed', res);
        }

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// get all

// Fetch the event categories matching the IDs in the array

async function findEventCatData(arrIds) {
    try {
        const eventCategory = await EventCategoryModel.findAll({
            where: {
                id: {
                    [Sequelize.Op.in]: arrIds,
                },
            },

            attributes: ['id', 'title'],
        });

        return eventCategory;
    } catch (err) {
        console.log(err);
        return [];
    }
}

exports.getAll = async (_, res) => {
    try {
        const result = await EventsModel.findAll();

        const eventCatArr = result.reduce((acc, cur) => {
            const catArr = cur.event_cat ? cur.event_cat.split(',') : [];
            catArr.forEach((catId) => {
                if (!acc.includes(catId)) {
                    acc.push(catId);
                }
            });
            return acc;
        }, []);

        const eventCategory = await findEventCatData(eventCatArr);

        const dataWithCat = result.map((item) => {
            const catIds = item.event_cat ? item.event_cat.split(',') : [];
            const catArrData = catIds.map((catId) =>
                eventCategory.find((cat) => cat.id === Number(catId))
            );
            return {
                ...item.dataValues,
                event_category: catArrData,
            };
        });

        return successResponse(200, 'OK', dataWithCat, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server side problem', res);
    }
};

// get one

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;
        const { status } = req.query;

        const whereCondition = {
            id,
        };

        if (status === 'active') {
            whereCondition.status = 1;
        }

        // Fetch the event entry
        const result = await EventsModel.findOne({
            where: whereCondition,
        });

        // If the event is not found, return a 404 response
        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO Active data found', res);
        }

        // Split the event_cat string into an array
        const eventCatArr = result.event_cat ? result.event_cat.split(',') : [];

        // Fetch the event categories matching the IDs in the array
        const eventCategory = await EventCategoryModel.findAll({
            where: {
                id: {
                    [Sequelize.Op.in]: eventCatArr,
                },
            },

            attributes: ['id', 'title'],
        });

        // Add the retrieved categories to the event result
        result.dataValues.event_category = eventCategory;

        // Return the success response with the event and its categories
        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

// update

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const {
            title,
            event_cat,
            date_type,
            event_date,
            event_date_time,
            short_description,
            description,
            status,
            location,
            order,
        } = req.body;

        console.log(req.body);

        const { files } = req;

        const isExist = await EventsModel.findOne({
            where: {
                id,
            },
        });

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No Data found with this id', res);
        }

        const event_img =
            files?.find((file) => file?.fieldname === 'event_img')?.filename || isExist?.event_img;

        const eventData = {
            title,
            event_cat,
            date_type,
            event_date: new Date(event_date),
            event_date_time,
            event_img,
            short_description,
            description,
            status,
            location,
            order,
        };
        console.log(eventData);
        const result = await EventsModel.update(eventData, {
            where: {
                id,
            },
        });

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Events updated failed', res);
        }

        return successResponse(201, 'OK', 'Update Successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

// delete

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await EventsModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Event modal deleted failed', res);
        }

        return successResponse(204, 'OK', 'Event deleted successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server side error', res);
    }
};
