const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
const db = require('../../../../config/database.config');
const errorResponse = require('../../../../utils/errorResponse');
const { expiryDateCalculate } = require('../../../../utils/expirayDateCalculate');
const successResponse = require('../../../../utils/successResponse');

const redirectURL = `${process.env.REACT_APP_API_BASE_URL}/payment/stripe/payment-status`;

const PaymentModel = db.model.user_payment;
const PackageModel = db.model.package;
const UserSubscription = db.model.userSubscription;
const UserModel = db.model.user;
const DonationModel = db.model.donation;

async function storePaymentData(paymentIntent, metadata) {
    try {
        const paymentData = {
            user_id: metadata.user_id,
            tran_date: new Date(),
            tran_id: paymentIntent.id,
            user_mobile: metadata.mobile,
            account_id: metadata.account_id || null, // Assuming you might want to include an account ID
            payment_user_name: `${metadata.f_name} ${metadata.l_name}`,
            payment_user_email: metadata.email,
            payment_location: metadata.payment_location || 'Unknown', // Adjust as needed
            total_amount: paymentIntent.amount / 100, // Stripe amount is in cents
            payment_fee: 0, // Stripe API doesn't return fee details in the payment intent
            store_amount: paymentIntent.amount / 100, // Assuming no fee
            currency: paymentIntent.currency,
            payment_package_id: metadata.package_id,
            campeign_id: metadata.campeign_id || null,
            card_type: metadata.payment_method || 'stripe',
            card_brand: 'stripe', // Stripe API doesn't return card brand in the payment intent
            status: paymentIntent.status === 'succeeded' ? 1 : 0,
        };

        const results = await PaymentModel.create(paymentData);
        return results;
    } catch (err) {
        throw new Error(err);
    }
}

async function storeSubscription(paymentIntent, metadata) {
    try {
        const packageData = await PackageModel.findOne({ where: { id: metadata?.package_id } });
        if (!packageData) {
            throw new Error('Package not found');
        }

        const subscriptionData = {
            user_id: metadata.user_id,
            subscription_type: 'membership Package',
            package_id: metadata.package_id,
            payment_id: metadata.payment_id,
            purchase_date: new Date(),
            expiry_date:
                (packageData.duration && expiryDateCalculate(packageData.duration)) || null,
            status: 1,
        };

        const result = await UserSubscription.create(subscriptionData);

        return result;
    } catch (err) {
        throw new Error(err);
    }
}

async function storeDonationData(paymentIntent, metadata) {
    try {
        const paymentData = {
            user_id: metadata.user_id,
            payment_id: metadata.payment_id,
            payment_tran_id: paymentIntent.id,
            donation_date: new Date(),
            tran_id: paymentIntent.id,
            f_name: metadata.f_name,
            l_name: metadata.l_name,
            mobile: metadata.mobile,
            email: metadata.email,
            donation_amount: paymentIntent.amount / 100, // Stripe amount is in cents
            currency: paymentIntent.currency,
            campeign_id: metadata.campeign_id || null,
            status: paymentIntent.status === 'succeeded' ? 1 : 0,
        };

        const results = await DonationModel.create(paymentData);
        return results;
    } catch (err) {
        throw new Error(err);
    }
}

exports.stripeWebHooks = async (req, res) => {
    try {
        const sig = req.headers['stripe-signature'];
        const event = stripe.webhooks.constructEvent(
            req.rawBody,
            sig,
            process.env.STRIPE_WEBHOOK_SECRET
        );
        console.log('event hook', event);

        // Handle the event
        switch (event.type) {
            case 'payment_intent.succeeded': {
                const paymentIntent = event.data.object;
                const { mobile } = paymentIntent.metadata;

                // Find user data based on mobile number
                const userData = await UserModel.findOne({ where: { mobile } });

                // Add user_id to metadata if user is found
                const metadata = {
                    ...paymentIntent.metadata,
                    user_id: userData?.id || null,
                };

                // Store payment data
                const paymentData = await storePaymentData(paymentIntent, metadata);

                // Add payment_id to metadata for donation data
                metadata.payment_id = paymentData.id;

                if (metadata.payment_type === 'package') {
                    await storeSubscription(paymentIntent, metadata);
                } else {
                    // Store donation data
                    await storeDonationData(paymentIntent, metadata);
                }

                successResponse(
                    200,
                    'SUCCESS',
                    'Payment and donation data stored successfully',
                    res
                );
                break;
            }

            // case 'payment_method.attached': {
            //     const paymentMethod = event.data.object;
            //     // Define and call a method to handle the successful attachment of a PaymentMethod
            //     // handlePaymentMethodAttached(paymentMethod);
            //     break;
            // }

            // Handle other event types
            default:
                errorResponse(500, 'ERROR', 'Unhandled event type', res);
                break;
        }
    } catch (err) {
        console.error('Webhook Error:', err.message);
        errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while processing the webhook',
            res
        );
    }
};
