const db = require('../../../config/database.config');
const errorResponse = require('../../../utils/errorResponse');
const successResponse = require('../../../utils/successResponse');

const { Op } = db.Sequelize;
const User = db.model.user;
const Role = db.model.role;
const RolePermission = db.model.rolePermission;
const UserDetails = db.model.user_details;

exports.findAll = async (req, res) => {
    try {
        const data = await User.findAll({
            include: [
                {
                    model: Role,
                    attributes: ['name', 'info'],
                },
            ],
        });

        return successResponse(200, 'OK', data, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while Finding User',
            res
        );
    }
};

exports.findAllCustomers = async (req, res) => {
    try {
        const data = await User.findAll({
            where: {
                role_id: 12,
            },
        });

        return successResponse(200, 'OK', data, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while Finding User',
            res
        );
    }
};

exports.findOne = async (req, res) => {
    try {
        const { id } = req.params;

        const userData = await User.findOne({
            where: {
                id,
            },
            include: [
                {
                    model: UserDetails,
                },
                {
                    model: Role,
                    attributes: ['name', 'info'],
                },
                {
                    model: RolePermission,
                    as: 'permissions',
                },
            ],
        });

        if (!userData) {
            return errorResponse(404, 'NOT FOUND', 'User not found', res);
        }

        return successResponse(200, 'OK', userData, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while Finding User',
            res
        );
    }
};

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const { user_details, ...userData } = req.body;
        console.log(userData);

        const isExist = await User.findByPk(id);

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'NO user found with this id', res);
        }

        if (!userData?.f_name && !userData?.role_id && !userData?.mobile) {
            return errorResponse(404, 'ERROR', 'Required Fields Cannot be Empty!!!', res);
        }

        const userObjData = {
            role_id: userData.role_id ? userData.role_id : 4,
            f_name: userData.f_name ?? null,
            l_name: userData.l_name ?? null,
            mobile: userData.mobile ?? null,
            email: userData.email ?? null,
            date_of_birth: userData.date_of_birth ?? null,
            contact_person: userData.contact_person ?? null,
            nid: userData.nid ?? null,
            credit_limit: userData.credit_limit ?? null,
            commission_rate: userData.commission_rate ?? null,
            address_1: userData.address_1 ?? null,
            address_2: userData.address_2 ?? null,
            remarks: userData.remarks ?? null,
            image: userData.image ?? isExist?.image,
            finger_print: userData.finger_print ?? null,
            status: userData.status,
        };

        const updateResult = await User.update(userObjData, {
            where: {
                id,
            },
        });

        if (!updateResult.includes(1)) {
            return errorResponse(404, 'FAILED', 'User Update Failed', res);
        }

        if (user_details) {
            const userDetailsData = {
                user_id: id,
                spouse_f_name: user_details?.spouse_f_name || null,
                spouse_l_name: user_details?.spouse_l_name || null,
                gender_id: user_details?.gender_id || null,
                marital_status_id: user_details?.marital_status_id || null,
                home_phone: user_details?.home_phone || null,
                work_phone: user_details?.work_phone || null,
                sons: Array.isArray(user_details?.sons)
                    ? user_details?.sons.join(',')
                    : user_details?.sons || null,
                daughters: Array.isArray(user_details?.daughters)
                    ? user_details?.daughters.join(',')
                    : user_details?.daughters || null,
                country_id: user_details?.country_id || null,
                state_id: user_details?.state_id || null,
                city_id: user_details?.city_id || null,
            };
            await UserDetails.update(userDetailsData, {
                where: {
                    id,
                },
            });
        }

        return successResponse(202, 'OK', 'User Update successfully', res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while Updating User',
            res
        );
    }
};

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const query = await User.findByPk(id);
        if (!query) {
            return errorResponse(404, 'NOT FOUND', 'User Not found', res);
        }

        await User.destroy({
            where: {
                id,
            },
        });

        return successResponse(204, 'OK', 'User Delete successfully', res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while Deleting the User.',
            res
        );
    }
};

exports.findAllByBirthdayRange = async (req, res) => {
    try {
        const dateRange = req.body;
        console.log(dateRange);

        const options = { month: '2-digit', day: '2-digit' };
        const startDate = new Date(dateRange.startDate);
        const startDateFormate = startDate.toLocaleDateString('en-US', options);
        const sqlStartDate = startDateFormate.replace('/', '-');

        const endDate = new Date(dateRange.endDate);
        const endDateFormate = endDate.toLocaleDateString('en-US', options);
        const sqlEndDate = endDateFormate.replace('/', '-');

        const targetRole = dateRange.role_id;

        console.log({
            sqlStartDate,
            sqlEndDate,
            targetRole,
        });

        if (targetRole) {
            // const likeStartDate = { [Op.like]: `%${sqlStartDate}` };
            // const likeEndDate = { [Op.like]: `%${sqlEndDate}` };

            const data = await User.findAll({
                where: {
                    // role_id: targetRole,
                    date_of_birth: {
                        // [Op.between]: ['2014-06-18', '2023-10-19'],
                        // [Op.like]: `${`2014-06-18`}`,
                        [Op.like]: `${sqlStartDate}`,
                    },
                },
            });

            successResponse(200, 'OK', data, res);
        } else {
            const data = await User.findAll({
                where: {
                    id: {
                        [Op.between]: [30, 27],
                    },
                },
            });

            successResponse(200, 'OK', data, res);
        }
    } catch (error) {
        console.error(error);
        res.status(500).json({ error: 'Internal server error' });
    }
};
exports.uploadImage = async (req, res) => {
    try {
        const imageFiles = req.file;

        // console.log(imageFiles);

        if (!imageFiles) {
            return errorResponse(400, 'ERROR', 'File Not Found', res);
        }

        return successResponse(200, 'OK', imageFiles, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while Finding Users By Date_of_Birth.',
            res
        );
    }
};

exports.findAllUsersByRoleId = async (req, res) => {
    try {
        const role_id = req.params.id;
        console.log(role_id);

        const allUsers = await User.findAll({
            where: {
                role_id,
            },
        });

        return successResponse(200, 'OK', allUsers, res);
    } catch (error) {
        return errorResponse(
            500,
            'ERROR',
            error.message || 'Some error occurred while Finding User',
            res
        );
    }
};
