const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');
const { booking_status } = require('../constant/booking.constant');

const bookingModel = db.model.booking;

exports.create = async (req, res) => {
    try {
        const { date, name, phone, email, tour_id, status } = req.body;

        const bookingData = {
            date,
            name,
            phone,
            email,
            tour_id,
            // status: booking_status.PENDING,
            status,
        };

        const result = await bookingModel.create(bookingData);
        // console.log('result of booking create =>', result);
        if (!result) {
            return errorResponse(404, 'FAILED', 'Booking creation failed', res);
        }

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server-side problem', res);
    }
};

// Get all bookings
exports.getAll = async (_, res) => {
    try {
        const result = await bookingModel.findAll();

        return successResponse(200, 'OK', result, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server-side problem', res);
    }
};

// Get one booking
exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await bookingModel.findOne({
            where: {
                id,
            },
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'No booking found with this ID', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (error) {
        return errorResponse(500, 'ERROR', error.message || 'There is a server-side problem', res);
    }
};

// Get active bookings
exports.getActiveData = async (_, res) => {
    try {
        const result = await bookingModel.findAll({
            where: {
                status: 1,
            },
        });

        if (!result || result.length === 0) {
            return errorResponse(404, 'NOT FOUND', 'No active bookings found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server-side error', res);
    }
};

// Update booking
exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const { date, name, phone, email, tour_id, status } = req.body;

        const isExist = await bookingModel.findOne({
            where: {
                id,
            },
        });

        if (!isExist) {
            return errorResponse(404, 'NOT FOUND', 'No booking found with this ID', res);
        }

        const bookingData = {
            date: date || isExist.date,
            name: name || isExist.name,
            phone: phone || isExist.phone,
            email: email || isExist.email,
            tour_id: tour_id || isExist.tour_id,
            // status: booking_status.PENDING,
            status: status || isExist.status,
        };

        const result = await bookingModel.update(bookingData, {
            where: {
                id,
            },
        });

        if (!result.includes(1)) {
            return errorResponse(404, 'FAILED', 'Booking update failed', res);
        }

        return successResponse(201, 'OK', 'Booking updated successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server-side problem', res);
    }
};

// Delete booking
exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await bookingModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'FAILED', 'Booking deletion failed', res);
        }

        return successResponse(204, 'OK', 'Booking deleted successfully', res);
    } catch (err) {
        return errorResponse(500, 'ERROR', 'There is a server-side error', res);
    }
};
