const { Op } = require('sequelize');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');
const { findCourseCatData } = require('./courses.controller');
const db = require('../../../../config/database.config');

const CoursesModel = db.model.courses;
const UserModel = db.model.user;
const CourseRegModel = db.model.courseRegistration;

/**
 *
 * @param cat_id -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Courses[] -- all active courses Data
 * * */

const getAllActiveCourseData = async (cat_id) => {
    try {
        const whereCondition = {
            status: 1,
        };
        // console.log(cat_id);

        if (cat_id) {
            whereCondition.course_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        const result = await CoursesModel.findAll({
            where: whereCondition,
            include: [
                {
                    model: UserModel,
                    as: 'instructor',
                    attributes: [
                        'id',
                        'f_name',
                        'l_name',
                        'email',
                        'mobile',
                        'image',
                        'role_id',
                        'remarks',
                    ],
                },
                {
                    model: CourseRegModel,
                },
            ],
        });
        console.log(result);
        return result;
    } catch (err) {
        console.log(err);
        throw new Error(err);
    }
};

/**
 * @param  limit -- required -- if limit passed it will limit the result
 * @param page -- required -- if page passed it will filter on that
 *
 * @param cat_id <Number> -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Courses[] -- all active courses Data
 * * */

const getCourseDataWithLimit = async (limit = 2, page = 1, cat_id) => {
    try {
        const offset = limit * (page - 1);

        const whereCondition = {
            status: 1,
        };

        if (cat_id) {
            whereCondition.course_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        const result = await CoursesModel.findAll({
            where: whereCondition,
            include: [
                {
                    model: UserModel,
                    as: 'instructor',
                    attributes: [
                        'id',
                        'f_name',
                        'l_name',
                        'email',
                        'mobile',
                        'image',
                        'role_id',
                        'remarks',
                    ],
                },
                {
                    model: CourseRegModel,
                },
            ],

            limit: Number(limit),
            offset: Number(offset),
        });

        const courseCatArr = result.reduce((acc, cur) => {
            const catArr = cur.course_cat ? cur.course_cat.split(',') : [];
            catArr.forEach((catId) => {
                if (!acc.includes(catId)) {
                    acc.push(catId);
                }
            });
            return acc;
        }, []);

        const courseCategory = await findCourseCatData(courseCatArr);

        const dataWithCat = result.map((item) => {
            const catIds = item.course_cat ? item.course_cat.split(',') : [];
            const catArrData = catIds.map((catId) =>
                courseCategory.find((cat) => cat.id === Number(catId))
            );
            return {
                ...item.dataValues,
                course_category: catArrData,
            };
        });

        const totalItems = await CoursesModel.count({
            where: {
                status: 1,
            },
        });
        const totalPages = Math.ceil(totalItems / limit);

        const hasNextPage = page < totalPages;

        return {
            data: dataWithCat,
            metaData: {
                hasNextPage,
                totalPages,
            },
        };
    } catch (err) {
        console.log(err);
        return false;
    }
};

const getActiveData = async (req, res) => {
    try {
        const { limit, page } = req.query;

        let result = null;

        if (limit && page) {
            result = await getCourseDataWithLimit(limit, page);
        } else {
            result = await getAllActiveCourseData();
        }

        // if (!result) {
        //   return errorResponse(404, "NOT FOUND", "NO Active data found", res);
        // }
        //
        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

module.exports = {
    getActiveData,
    getCourseDataWithLimit,
    getAllActiveCourseData,
};
