const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const PagesModel = db.model.pages;

const LeftMenuPage = db.model.left_menu_page;

// single images upload to website folder
exports.uploadImages = async (req, res) => {
    try {
        const files = req?.files;
        console.log(files);
        if (!files) {
            return errorResponse(404, 'FAILED', 'No file found', res);
        }

        return successResponse(200, 'OK', files, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side error', res);
    }
};

exports.create = async (req, res) => {
    try {
        const { title, url, content, page_banner, left_menu } = req.body;

        const result = await PagesModel.create({
            title,
            url,
            content,
            page_banner,
        });

        if (!result) {
            return errorResponse(404, 'FAILED', 'Page created failed', res);
        }

        Promise.all(
            left_menu.map(async (menu) =>
                LeftMenuPage.create({
                    name: menu.name,
                    url: menu.url,
                    page_id: result.id,
                })
            )
        );

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.getAll = async (_, res) => {
    try {
        const result = await PagesModel.findAll();

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.getActiveData = async (_, res) => {
    try {
        const result = await PagesModel.findAll({
            where: {
                status: 1,
            },
        });

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.getContent = async (req, res) => {
    try {
        const { url } = req.query;
        // console.log(url);
        const result = await PagesModel.findOne({
            where: {
                url,
                status: 1,
            },
            include: [
                {
                    model: LeftMenuPage,
                    as: 'left_menu',
                },
            ],
        });

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.getOne = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await PagesModel.findOne({
            where: {
                id,
            },
            include: [
                {
                    model: LeftMenuPage,
                    as: 'left_menu',
                },
            ],
        });

        if (!result) {
            return errorResponse(404, 'NOT FOUND', 'NO data found', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.update = async (req, res) => {
    try {
        const { id } = req.params;

        const { title, url, content, page_banner, left_menu } = req.body;

        const result = await PagesModel.update(
            {
                title,
                url,
                content,
                page_banner,
            },
            {
                where: {
                    id,
                },
            }
        );

        if (!result?.includes(1)) {
            return errorResponse(404, 'FAILED', 'Page update failed', res);
        }

        await LeftMenuPage.destroy({
            where: {
                page_id: id,
            },
        });

        Promise.all(
            left_menu.map(async (menu) =>
                LeftMenuPage.create({
                    name: menu.name,
                    url: menu.url,
                    page_id: id,
                })
            )
        );

        return successResponse(201, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const result = await PagesModel.destroy({
            where: {
                id,
            },
        });

        if (result !== 1) {
            return errorResponse(404, 'NOT FOUND', 'Delete Failed', res);
        }

        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err?.message || 'There is a server side problem', res);
    }
};
