const { Op } = require('sequelize');
const db = require('../../../../config/database.config');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');

// dbs

const MenuGroup = db.model.menuGroup;

const MenuItems = db.model.menuItems;

// functions

exports.create = async (req, res) => {
    try {
        const reqData = req.body;
        console.log(reqData);

        const { menu_level, url_slug, menu_order, menu_icon, parent_menu_id, menu_group_id } =
            reqData;

        if (!menu_level || !menu_group_id) {
            return errorResponse(403, 'INVALID', 'Invalid menu input', res);
        }

        const storingObject = {
            menu_level,
            url_slug,
            menu_order,
            menu_icon,
            parent_menu_id,
            menu_group_id,
        };

        const results = await MenuItems.create(storingObject);

        if (!results) {
            return errorResponse(404, 'FAILED', 'Failed to create new menu Group', res);
        }

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error creating the menu group',
            res
        );
    }
};

// find all the menu groups

exports.findAll = async (req, res) => {
    try {
        const results = await MenuItems.findAll();

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error finding the menu group',
            res
        );
    }
};

// get menu with sub menu

exports.getMenuWithSubMenu = async (req, res) => {
    try {
        const { group_id } = req.params;

        const results = await MenuItems.findAll({
            where: {
                parent_menu_id: { [Op.or]: [0, null] },
                menu_group_id: group_id,
            },
            include: [
                {
                    model: MenuItems,
                    as: 'subMenuItems',
                    order: [['menu_order', 'ASC']],
                    separate: true,
                },
            ],
            order: [['menu_order', 'ASC']],
        });

        return successResponse(200, 'OK', results, res);
    } catch (error) {
        return errorResponse(
            500,
            'ERROR',
            error.message || 'There was a server-side error finding the menu group',
            res
        );
    }
};

exports.update = async (req, res) => {
    try {
        const { id } = req.params;
        const reqData = req.body;

        const { menu_level, url_slug, menu_order, menu_icon, parent_menu_id, menu_group_id } =
            reqData;

        if (!menu_level || !menu_group_id) {
            return errorResponse(403, 'INVALID', 'Invalid menu input', res);
        }

        const storingObject = {
            menu_level,
            url_slug,
            menu_order,
            menu_icon,
            parent_menu_id,
            menu_group_id,
        };

        const results = await MenuItems.update(storingObject, {
            where: {
                id,
            },
        });

        if (!results.includes(1)) {
            return errorResponse(404, 'FAILED', 'Failed to updating new menu Item', res);
        }

        return successResponse(201, 'OK', 'Menu Items updated Successfully', res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error updating the menu group',
            res
        );
    }
};

// delete single menu group

exports.delete = async (req, res) => {
    try {
        const { id } = req.params;

        const results = await MenuItems.destroy({
            where: {
                id,
            },
        });

        if (results !== 1) {
            return errorResponse(404, 'FAILED', 'Deleting menu group failed', res);
        }

        return successResponse(201, 'OK', results, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'There was a server side error finding the menu group',
            res
        );
    }
};
