const { Op } = require('sequelize');

const db = require('../../../../config/database.config');

const errorResponse = require('../../../../utils/errorResponse');

const successResponse = require('../../../../utils/successResponse');

const CampeignsModel = db.model.campeigns;
const { findcampeignCatData } = require('./campeign.controller');

/**
 *
 * @param cat_id -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Campeigns[] -- all active campeigns Data
 * * */

const getAllActiveCampeignData = async (cat_id) => {
    try {
        const whereCondition = {
            status: 1,
        };
        console.log(cat_id);

        if (cat_id) {
            whereCondition.campeign_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        const result = await CampeignsModel.findAll({
            where: whereCondition,
        });
        console.log(result);
        return result;
    } catch (err) {
        console.log(err);
        return null;
    }
};

/**
 * @param  limit -- required -- if limit passed it will limit the result
 * @param page -- required -- if page passed it will filter on that
 *
 * @param cat_id : number -- optional -- if cat id passed it will filter on that
 *
 * @returns -- Campeigns[] -- all active campeigns Data
 * * */

const getCampeignDataWithLimit = async (limit = 2, page = 1, cat_id) => {
    try {
        const offset = limit * (page - 1);

        const whereCondition = {
            status: 1,
        };

        if (cat_id) {
            whereCondition.campeign_cat = {
                [Op.like]: `%${cat_id}%`,
            };
        }

        // find all the campeigns data
        const result = await CampeignsModel.findAll({
            where: whereCondition,
            limit: Number(limit),
            offset: Number(offset),
        });

        // take all the category ids from the campeigns
        const campeignCatArr = result.reduce((acc, cur) => {
            const catArr = cur.campeign_cat ? cur.campeign_cat.split(',') : [];
            catArr.forEach((catId) => {
                if (!acc.includes(catId)) {
                    acc.push(catId);
                }
            });
            return acc;
        }, []);

        // fetch only campeign category data with unique ids
        const campeignCategory = await findcampeignCatData(campeignCatArr);

        // destribute the category data in campeign
        const dataWithCat = result.map((item) => {
            const catIds = item.campeign_cat ? item.campeign_cat.split(',') : [];
            const catArrData = catIds.map((catId) =>
                campeignCategory.find((cat) => cat.id === Number(catId))
            );
            return {
                ...item.dataValues,
                campeign_category: catArrData,
                donation_collected: 100,
            };
        });

        const totalItems = await CampeignsModel.count({
            where: {
                status: 1,
            },
        });
        const totalPages = Math.ceil(totalItems / limit);

        const hasNextPage = page < totalPages;

        return {
            data: dataWithCat,
            metaData: {
                hasNextPage,
                totalPages,
            },
        };
    } catch (err) {
        console.log(err);
        return null;
    }
};

const getActiveData = async (req, res) => {
    try {
        const { limit, page } = req.query;

        let result = null;

        if (limit && page) {
            result = await getCampeignDataWithLimit(limit, page);
        } else {
            result = await getAllActiveCampeignData();
        }

        // if (!result) {
        //   return errorResponse(404, "NOT FOUND", "NO Active data found", res);
        // }
        //
        return successResponse(200, 'OK', result, res);
    } catch (err) {
        return errorResponse(500, 'ERROR', err.message || 'There was a server side Error', res);
    }
};

module.exports = {
    getActiveData,
    getCampeignDataWithLimit,
    getAllActiveCampeignData,
};
