const Paypal = require('@paypal/checkout-server-sdk');

const fetch = require('node-fetch');
const db = require('../../../../config/database.config');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');
const { paypalClient, generatePaypalAccessToken } = require('./paypalConfig');

const base = 'https://api-m.sandbox.paypal.com';

// dbs

const PaymentModel = db.model.user_payment;
const UserSubscription = db.model.userSubscription;
const PackageModel = db.model.package;
const DonationModel = db.model.donation;

async function storePaymentData(result, bodyData) {
    try {
        // console.log('paypal result', result);
        if (result?.status === 'COMPLETED') {
            const { payer, purchase_units, payment_source } = result;

            const capture = purchase_units[0].payments.captures[0];
            const paymentData = {
                user_id: bodyData?.user_id, // Pass the user ID from your application
                tran_date: capture?.create_time,
                tran_id: capture?.id,
                user_mobile: bodyData.mobile,
                account_id: payment_source?.paypal?.account_id,
                payment_user_name: `${payer.name.given_name} ${payer.name.surname ?? ''}`,
                payment_user_email: payer.email_address,
                payment_location: `${purchase_units[0].shipping.address.admin_area_2}, ${purchase_units[0].shipping.address.admin_area_1}`,
                total_amount: capture?.amount?.value,
                payment_fee: capture?.seller_receivable_breakdown?.paypal_fee?.value,
                store_amount: capture?.seller_receivable_breakdown?.net_amount?.value,
                currency: capture?.amount?.currency_code,
                // payment_package_id: packageId,
                campeign_id: bodyData?.campeign_id || null,
                card_type: bodyData?.payment_method,
                card_brand: bodyData?.payment_method,
                status: result?.status === 'COMPLETED' ? 1 : 0,
            };

            const results = await PaymentModel.create(paymentData);
            return results;
        }
        return null;
    } catch (err) {
        throw new Error(err);
    }
}

// stored donation data

async function storeDonationData(result, bodyData) {
    try {
        if (result?.status === 'COMPLETED') {
            const { purchase_units, payment_source } = result;

            const capture = purchase_units[0].payments.captures[0];
            const paymentData = {
                user_id: bodyData?.user_id,
                payment_id: bodyData?.payment_id,
                payment_tran_id: capture?.id,
                donation_date: new Date(),
                tran_id: capture?.id,
                f_name: bodyData.f_name,
                l_name: bodyData.l_name,
                mobile: bodyData.mobile,
                email: bodyData.email,
                donation_amount: capture.amount.value,
                currency: capture.amount.currency_code,
                campeign_id: bodyData?.campeign_id || null,
                status: result?.status === 'COMPLETED' ? 1 : 0,
            };

            const results = await DonationModel.create(paymentData);
            return results;
        }
        return null;
    } catch (err) {
        throw new Error(err);
    }
}

// exports.captureDonationPayment = async (req, res) => {
//     try {
//         const { order_id, f_name, l_name, mobile, email, donation_amount, payment_method } =
//             req.body;
//         const { user } = req;
//         console.log(req.body);
//         req.body.user_id = user?.id ? user?.id : null;
//
//         if (!order_id) {
//             return errorResponse(400, 'ERROR', 'Order id is required', res);
//         }
//
//         const PaypalClient = paypalClient();
//         const request = new Paypal.orders.OrdersCaptureRequest(order_id);
//         request.requestBody({});
//         const response = await PaypalClient.execute(request);
//         // console.log('paypal capture', response);
//         if (response.statusCode !== 201) {
//             return errorResponse(400, 'ERROR', 'Payment Failed. Please try again', res);
//         }
//         const storedData = await storePaymentData(response.result, req.body);
//
//         if (!storedData) {
//             return errorResponse(
//                 500,
//                 'ERROR',
//                 'Payment successfull but data stored in database failed. Please check the server',
//                 res
//             );
//         }
//
//         return successResponse(201, 'OK', storedData, res);
//     } catch (err) {
//         console.log(err);
//         return errorResponse(
//             500,
//             'ERROR',
//             err.message || 'Some error occurred while creating order id',
//             res
//         );
//     }
// };

// capture function manually with apis

exports.captureDonationPayment = async (req, res) => {
    try {
        const { order_id } = req.body;
        const { user } = req;
        req.body.user_id = user?.id ? user?.id : null;

        if (!order_id) {
            return errorResponse(400, 'ERROR', 'Order id is required', res);
        }

        const accessToken = await generatePaypalAccessToken();
        const url = `${base}/v2/checkout/orders/${order_id}/capture`;

        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                Authorization: `Bearer ${accessToken}`,
                // Uncomment one of these to force an error for negative testing (in sandbox mode only).
                // Documentation:
                // https://developer.paypal.com/tools/sandbox/negative-testing/request-headers/
                // "PayPal-Mock-Response": '{"mock_application_codes": "INSTRUMENT_DECLINED"}'
                // "PayPal-Mock-Response": '{"mock_application_codes": "TRANSACTION_REFUSED"}'
                // "PayPal-Mock-Response": '{"mock_application_codes": "INTERNAL_SERVER_ERROR"}'
            },
        });

        const parseResponse = await response.json();
        console.log('paypal capture', parseResponse);

        if (response.status !== 201) {
            return errorResponse(400, 'ERROR', 'Payment Failed. Please try again', res);
        }
        const storedData = await storePaymentData(parseResponse, req.body);

        if (!storedData) {
            return errorResponse(
                500,
                'ERROR',
                'Payment successfull but data stored in database failed. Please check the server',
                res
            );
        }

        // stored donaton data
        req.body.payment_id = storedData?.id;

        const storeDonatio = await storeDonationData(parseResponse, req.body);

        if (!storeDonatio) {
            return errorResponse(
                500,
                'ERROR',
                'Payment successfull but data stored in donation database failed. Please check the server',
                res
            );
        }

        return successResponse(201, 'OK', storedData, res);
    } catch (err) {
        console.log(err);
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while creating order id',
            res
        );
    }
};
