const Paypal = require('@paypal/checkout-server-sdk');

const db = require('../../../../config/database.config');
const errorResponse = require('../../../../utils/errorResponse');
const successResponse = require('../../../../utils/successResponse');
const { paypalClient, generatePaypalAccessToken } = require('./paypalConfig');
const { expiryDateCalculate } = require('../../../../utils/expirayDateCalculate');

// dbs

const PaymentModel = db.model.user_payment;
const UserSubscription = db.model.userSubscription;
const PackageModel = db.model.package;

const base = 'https://api-m.sandbox.paypal.com';

exports.createPaypalPaymentOrder = async (req, res) => {
    try {
        const { order_price } = req.body;
        if (!order_price) {
            return errorResponse(400, 'ERROR', 'Order price is required', res);
        }

        const PaypalClientInt = paypalClient();
        const request = new Paypal.orders.OrdersCreateRequest();
        request.headers.prefer = 'return=representation';
        request.requestBody({
            intent: 'CAPTURE',
            purchase_units: [
                {
                    amount: {
                        currency_code: 'USD',
                        value: `${order_price}`,
                    },
                },
            ],
        });

        const response = await PaypalClientInt.execute(request);
        console.log('paypal create', response);

        if (response.statusCode !== 201) {
            console.log(response);
            return errorResponse(400, 'ERROR', 'Error occured while creating order id', res);
        }

        successResponse(200, 'OK', response.result, res);
    } catch (err) {
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while creating order id',
            res
        );
    }
};

async function storePaymentData(result, userId, packageId) {
    try {
        // console.log('paypal result', result);
        if (result.status === 'COMPLETED') {
            const { payer, purchase_units, payment_source } = result;
            console.log(payer);

            const capture = purchase_units[0].payments.captures[0];
            const paymentData = {
                user_id: userId, // Pass the user ID from your application
                tran_date: capture.create_time,
                tran_id: capture.id,
                account_id: payment_source?.paypal?.account_id,
                payment_user_name: `${payer.name.given_name} ${payer.name.surname ?? ''}`,
                payment_user_email: payer.email_address,
                payment_location: `${purchase_units[0].shipping.address.admin_area_2}, ${purchase_units[0].shipping.address.admin_area_1}`,
                total_amount: capture.amount.value,
                payment_fee: capture?.seller_receivable_breakdown?.paypal_fee?.value,
                store_amount: capture.seller_receivable_breakdown.net_amount.value,
                currency: capture.amount.currency_code,
                payment_package_id: packageId,
                card_type: 'PayPal', // Since it's a PayPal payment
                card_brand: 'PayPal', // Since it's a PayPal payment
                status: result?.status === 'COMPLETED' ? 1 : 0,
            };

            const results = await PaymentModel.create(paymentData);
            return results;
        }
        return null;
    } catch (err) {
        throw new Error(err);
    }
}

// store subscription after payment
async function storeSubscription(userId, packageId, payment_id) {
    try {
        const packageData = PackageModel.findOne({ where: { id: packageId } });
        if (!packageData) {
            throw new Error('Package not found');
        }

        const subscriptionData = {
            user_id: userId,
            subscription_type: 'membership Package',
            package_id: packageId,
            payment_id,
            purchase_date: new Date(),
            expiry_date: expiryDateCalculate(packageData.duration),
            status: 1,
        };

        const result = await UserSubscription.create(subscriptionData);

        return result;
    } catch (err) {
        throw new Error(err);
    }
}

exports.capturePackagePaymentWithPaypal = async (req, res) => {
    try {
        const { orderId, packageId } = req.body;
        const { user } = req;

        if (!orderId) {
            return errorResponse(400, 'ERROR', 'Order id is required', res);
        }

        if (!packageId) {
            return errorResponse(400, 'ERROR', 'Please choose a membership package', res);
        }

        const accessToken = await generatePaypalAccessToken();
        const url = `${base}/v2/checkout/orders/${orderId}/capture`;

        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                Authorization: `Bearer ${accessToken}`,
                // Uncomment one of these to force an error for negative testing (in sandbox mode only).
                // Documentation:
                // https://developer.paypal.com/tools/sandbox/negative-testing/request-headers/
                // "PayPal-Mock-Response": '{"mock_application_codes": "INSTRUMENT_DECLINED"}'
                // "PayPal-Mock-Response": '{"mock_application_codes": "TRANSACTION_REFUSED"}'
                // "PayPal-Mock-Response": '{"mock_application_codes": "INTERNAL_SERVER_ERROR"}'
            },
        });

        const parseResponse = await response.json();

        if (response.status !== 201) {
            return errorResponse(400, 'ERROR', 'Payment Failed. Please try again', res);
        }

        // const PaypalClient = paypalClient();
        // const request = new Paypal.orders.OrdersCaptureRequest(orderId);
        // request.requestBody({});
        // const response = await PaypalClient.execute(request);
        // // console.log('paypal capture', response);
        // if (response.statusCode !== 201) {
        //     return errorResponse(400, 'ERROR', 'Payment Failed. Please try again', res);
        // }
        const storedData = await storePaymentData(parseResponse, user.id, packageId);

        if (!storedData) {
            return errorResponse(
                500,
                'ERROR',
                'Payment successfull but data stored in database failed. Please check the server',
                res
            );
        }

        await storeSubscription(user.id, packageId, storedData.id);
        return successResponse(201, 'OK', storedData, res);
    } catch (err) {
        console.log(err);
        return errorResponse(
            500,
            'ERROR',
            err.message || 'Some error occurred while creating order id',
            res
        );
    }
};
